/* 
 * The MIT License (MIT)
 *
 * Copyright (c) 2019 Ha Thach (tinyusb.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "b-em.h"
#include "pico/util/queue.h"
//#include "bsp/board.h"
#include "tusb.h"
#include "usb_host_hid.h"

#if CFG_TUH_HID
#define MAX_REPORT  4

static uint8_t const keycode2ascii[128][2] =  { HID_KEYCODE_TO_ASCII };

// Each HID instance can has multiple reports
static uint8_t _report_count[CFG_TUH_HID];
static tuh_hid_report_info_t _report_info_arr[CFG_TUH_HID][MAX_REPORT];

static void process_kbd_report(hid_keyboard_report_t const *report);

static uint8_t dev_addr_store;
static uint8_t instance_store;
static bool kbd_mounted = false;


//--------------------------------------------------------------------+
// TinyUSB Callbacks
//--------------------------------------------------------------------+

// Invoked when device with hid interface is mounted
// Report descriptor is also available for use. tuh_hid_parse_report_descriptor()
// can be used to parse common/simple enough descriptor.
void tuh_hid_mount_cb(uint8_t dev_addr, uint8_t instance, uint8_t const* desc_report, uint16_t desc_len)
{
  log_info("HID device address = %d, instance = %d is mounted\r\n", dev_addr, instance);

  // Interface protocol
  const char* protocol_str[] = { "None", "Keyboard", "Mouse" }; // hid_protocol_type_t
  uint8_t const interface_protocol = tuh_hid_interface_protocol(dev_addr, instance);

  // Parse report descriptor with built-in parser
  _report_count[instance] = tuh_hid_parse_report_descriptor(_report_info_arr[instance], MAX_REPORT, desc_report, desc_len);
  log_info("HID has %u reports and interface protocol = %s\r\n", _report_count[instance], protocol_str[interface_protocol]);
  if (interface_protocol==1) // Keyboard
  {
	  kbd_mounted = true;
	  dev_addr_store = dev_addr;
	  instance_store = instance;
  }
  
  // request to receive report
  // tuh_hid_report_received_cb() will be invoked when report is available
  if ( !tuh_hid_receive_report(dev_addr, instance) )
  {
    log_info("Mount Error: cannot request to receive report\r\n");
  }
}

// Invoked when device with hid interface is un-mounted
void tuh_hid_umount_cb(uint8_t dev_addr, uint8_t instance)
{
  log_info("HID device address = %d, instance = %d is unmounted\r\n", dev_addr, instance);
  if (kbd_mounted && dev_addr_store == dev_addr && instance_store==instance)
	  kbd_mounted = false;
}

// Invoked when received report from device via interrupt endpoint
void tuh_hid_report_received_cb(uint8_t dev_addr, uint8_t instance, uint8_t const* report, uint16_t len)
{
  (void) dev_addr;

  uint8_t const rpt_count = _report_count[instance];
  tuh_hid_report_info_t* rpt_info_arr = _report_info_arr[instance];
  tuh_hid_report_info_t* rpt_info = NULL;

  if ( rpt_count == 1 && rpt_info_arr[0].report_id == 0)
  {
    // Simple report without report ID as 1st byte
    rpt_info = &rpt_info_arr[0];
  }else
  {
    // Composite report, 1st byte is report ID, data starts from 2nd byte
    uint8_t const rpt_id = report[0];

    // Find report id in the array
    for(uint8_t i=0; i<rpt_count; i++)
    {
      if (rpt_id == rpt_info_arr[i].report_id )
      {
        rpt_info = &rpt_info_arr[i];
        break;
      }
    }

    report++;
    len--;
  }

  if (!rpt_info)
  {
    log_info("Couldn't find the report info for this report !\r\n");
    return;
  }

  if ( rpt_info->usage_page == HID_USAGE_PAGE_DESKTOP )
  {
    switch (rpt_info->usage)
    {
      case HID_USAGE_DESKTOP_KEYBOARD:
        TU_LOG1("HID receive keyboard report\r\n");
        // Assume keyboard follow boot report layout
        process_kbd_report( (hid_keyboard_report_t const*) report );
      break;


      default: break;
    }
  }
  
    // continue to request to receive report
  if ( !tuh_hid_receive_report(dev_addr, instance) )
  {
    log_info("Process Error: cannot request to receive report\r\n");
  }
}
#endif

//--------------------------------------------------------------------+
// USB Keyboard
//--------------------------------------------------------------------+
#if CFG_TUH_HID


queue_t kb_queue;


inline bool find_key_in_report(hid_keyboard_report_t const *p_report, uint8_t keycode)
{
  for(uint8_t i = 0; i < 6; i++)
  {
    if (p_report->keycode[i] == keycode)  return true;
  }

  return false;
}

static inline uint8_t keycode_to_ascii(uint8_t modifier, uint8_t keycode)
{
  return keycode > 128 ? 0 :
    keycode2ascii [keycode][modifier & (KEYBOARD_MODIFIER_LEFTSHIFT | KEYBOARD_MODIFIER_RIGHTSHIFT) ? 1 : 0];
}

static void push_key_event(bool down, int scancode) {
    uint32_t val = (down << 16) | scancode;
//	log_info("Add key: %lx",val);
    queue_try_add(&kb_queue, &val);
}

static void process_kbd_report(hid_keyboard_report_t const *report)
{
	static uint8_t last_modifiers;
	static hid_keyboard_report_t prev_keyboard_report;
	static hid_keyboard_report_t keyboard_report;
	
	
	prev_keyboard_report = keyboard_report;
	keyboard_report = *report;
	
//	log_info("Key report. Mod=%x , Codes=%d %d %d %d %d %d", keyboard_report.modifier, 
//	keyboard_report.keycode[0], keyboard_report.keycode[1], keyboard_report.keycode[2],
//	keyboard_report.keycode[3], keyboard_report.keycode[4], keyboard_report.keycode[5]
//	);
	
	uint8_t changed_modifiers = keyboard_report.modifier ^ last_modifiers;
	uint i=0;
	while (changed_modifiers && i < 8) {
		if (changed_modifiers & 1) {
			push_key_event(!(last_modifiers & 1), 224 + i);
		}
		changed_modifiers >>= 1;
		last_modifiers >>= 1;
		i++;
	}
	last_modifiers = keyboard_report.modifier;
	// I assume it's possible to have up to 6 keypress events per report?
	for(i = 0; i < 6; i++)
	{
		// Check for key presses
		if (keyboard_report.keycode[i]) {
			// If not in prev report then it is newly pressed
			if (!find_key_in_report(&prev_keyboard_report, keyboard_report.keycode[i])) {
				push_key_event(true, keyboard_report.keycode[i]);
			}
		}
		// Check for key depresses (i.e. was present in prev report but not here)
		if (prev_keyboard_report.keycode[i]) {
			// If not present in the current report then depressed
			if (!find_key_in_report(&keyboard_report, prev_keyboard_report.keycode[i]))
			{
				push_key_event(false, prev_keyboard_report.keycode[i]);
			}
		}
	}
}

void set_leds(const uint8_t leds) {
	if (kbd_mounted)
	{		
		// To set the LED lamps, the driver sends a SetReport request to the device using a standard USB Setup Transaction, with a one-byte data stage. 
		// The setup packet's request type should contain 0x21, the request code for SetReport is 0x09. 
		// The value field of the setup packet contains the report ID in the low byte, which should be zero. 
		// The high byte contains the report type, which should be 0x02 to indicate an output report, or a report that is being sent from the software to the hardware. 
		// The index field should contain the interface number of the USB keyboard, which is the number present in the interface descriptor which indicated this device was a USB keyboard at all. 
		// The data stage should be 1 byte, which is a bitfield.
		uint8_t temp;
		temp = leds>>5;
		log_info("Set LEDs: %x %x",leds, temp);
		// uint8_t dev_addr, uint8_t instance, uint8_t report_id, uint8_t report_type, void* report, uint16_t len
		tuh_hid_set_report(dev_addr_store, instance_store, 0, HID_REPORT_TYPE_OUTPUT, &temp, 1);
	}
};

bool get_kb_event(struct kb_event *event) {
    uint32_t data = 0;
    if (queue_try_remove(&kb_queue, &data)) {
//		log_info("Got key: %lx",data);
        event->down = data > 65535;
        event->scancode = (uint16_t)data;
        return true;
    }
    return false;
}

#endif



void usb_host_hid_init() {
#if CFG_TUH_HID
    queue_init(&kb_queue, sizeof(uint32_t), 4);
#endif
    tusb_init();
}

void usb_host_hid_poll() {
    tuh_task();
}
