/*
 * hat.c Driver for hat display
 */

#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/delay.h>
#include <avr/sleep.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <stdio.h>
#include <string.h>
#include <math.h>


#include "hatdefs.h"
#include "draw.h"
#include "utils.h"
#include "menu.h"





#define COURTL 4
#define COURTR 16
#define BATH 3


BYTE cyclefinished;
int16_t frames;


int8_t	xdir;
int8_t  ydir;
int8_t  xpos;
int8_t  ypos;
BYTE  lscore;
BYTE  rscore;
int8_t  lbath;
int8_t  rbath;
int8_t pongpause;
int8_t invadervar;
int8_t invaderframecount;
int8_t invaderthresh;


#include "analog.c"


void analogon(void)
{
	analoglevel=0;
	ADMUX=_BV(REFS0)+ANALOG_INPUT;
	ADCSRA=_BV(ADEN)|_BV(ADSC)|_BV(ADIE)|_BV(ADPS2)|_BV(ADPS1);
}

void analogoff(void)
{
	analoglevel=0;
	ADCSRA=0;
}

void readstring(uint16_t id, BYTE both)
{
	char* p;
	eeprom_read_block(display_string[1],(void*)(3+id*(uint16_t)DISPLAYSTRINGLEN),DISPLAYSTRINGLEN);
	p=strchr(display_string[1],92);
	if (p!=NULL)
		*p=0;
	p=strchr(display_string[1],0xff);
	if (p!=NULL)
		*p=0;
	if (both)
		memcpy(&(display_string[0][0]),&(display_string[1][0]),DISPLAYSTRINGLEN);
}

void initstring(void)
{
	string_ptr=0;
	string_index=0;
	string_cycle=0;
	char_col=0;
	frames=0;
	clearbuf();
	clearnextbuf();
	if (analogontext)
		analogon();
	else
		analogoff();
}




BYTE testsw(void)
{
	static BYTE oldsw;
	static uint16_t repcount;
	BYTE newsw;
	BYTE rval;
	
	newsw=PORTINPUT & INPUTBITS;
	rval =0;
	if (oldsw!=newsw)
	{
		oldsw=oldsw^newsw;
		if (oldsw & _BV(SW1))
			rval += (newsw & _BV(SW1)) ? SW1_UP : SW1_DOWN ;
		if (oldsw & _BV(SW2))
			rval += (newsw & _BV(SW2)) ? SW2_UP : SW2_DOWN ;
		if (oldsw & _BV(SW3))
			rval += (newsw & _BV(SW3)) ? SW3_UP : SW3_DOWN ;
		oldsw=newsw;
			repcount=0;
		return (rval);
	}
	else
	{
		if (newsw!=INPUTBITS)
		{
			repcount++;
			if (repcount==REPEATDELAY)
			{
				rval += (newsw & _BV(SW1)) ? 0: SW1_REPEAT ;
				rval +=(newsw & _BV(SW2)) ? 0: SW2_REPEAT;
			}
			else
			if (repcount>REPEATDELAY)
				if ((repcount-REPEATDELAY)%REPEATRATE==0)
				{
					rval += (newsw & _BV(SW1)) ? 0: SW1_REPEAT;
					rval +=(newsw & _BV(SW2)) ? 0: SW2_REPEAT;
				}
		}
		return(rval);
	}

}


void ioinit (void)
{
	
	count=0;
	framecount=0;
	active_buf=0;
	next_buf=0;
	
	eeprom_read_block(&analogdivider,(void*)(0),1);
	analogdivider = analogdivider - 48 + ANALOGDIVIDEROFFSET;
	
	setrand();
	initstring();
	
	// Set timing for interrupts
	OCR1A=CLOCK/(FPS * NROWS);
	// Set timer to CTC mode clocked at system clock frequency
	TCCR1B=_BV(WGM12)|_BV(CS10);
	// Enable interrupts on timer 1
	TIMSK=_BV(OCIE1A);
	
	// Setup serial output port
	PORTSERIAL=_BV(CK_BUF0);
	
	DDRSERIAL= OUTPUTBITS;
	
	//Setup analog input
	analogon();
	
	displays = 3;
	
	setupdisplays((displays&1),(displays&2));
	
	
	set_sleep_mode(SLEEP_MODE_IDLE);
	
	// Enable system interrupts
	sei();
	testsw();
}

void iteratesoundsquare(void)
{
	write_buf=1-next_buf;
	clearbuf();
	displayanalogsquare();
	next_buf=1-next_buf;
	frames++;
	if (frames>3000)
		cyclefinished=1;
}

void iteratetextscroll(void)
{
	BYTE i;
	write_buf=1-next_buf;
	BYTE instringcycle;


	if((framecount%10)==0)
	{
		copybuf();
		scrollleft();
		displayanaloglevel();
		i=(BYTE) display_string[string_index][string_ptr];
		if (i==(BYTE)0)
		{
			if ((mode==CYCLICTEXTMODE)&&(string_index!=0))
			{
				instringcycle=string_cycle;
				do {
					string_cycle=(string_cycle+(BYTE)1)%NSTRINGS;
					readstring(string_cycle,0);
				} while ((string_cycle!=instringcycle)&&(display_string[1][0]==0));
			}
			string_ptr=(BYTE)0;
			if (string_index==1)
				{
					frames++;
					if (frames > 2)
						cyclefinished=1;
				}
			string_index=(BYTE)1;
			i=(BYTE) display_string[string_index][string_ptr];
		}
		if (i!=0)
			if (rchar(i)==0)
				string_ptr++;
		next_buf=1-next_buf;
	}
	else
	{
		displayanaloglevel();
	}

}



void initpongball(void)
{
	updaterand();
	xdir = (rand & 1) ? 1 : -1;
	updaterand();
	ydir = (rand & 1) ? 1 : -1;
	updaterand();
	updaterand();
	xpos=(NCOLS/2)-1+(rand & 3);
	updaterand();
	updaterand();
	ypos=(NROWS/2)-1+(rand & 3);
	lbath=NROWS/2-1;
	rbath=lbath;
	pongpause=10;
}


void initpong(void)
{
	initpongball();
	lscore=0;
	rscore=0;
	frames=0;
}

void iteratepong(void)
{
	int8_t i;
	
	write_buf=1-next_buf;
	clearbuf();
	drawhline(COURTL, COURTR,0);
	drawhline(COURTL, COURTR,NROWS-1);
	if (xdir<0)
	{
		updaterand();
		updaterand();
		updaterand();
		updaterand();
		if ((rand & 15)<2)
		{
			i=ypos+ydir-(lbath+BATH/2);
			if (i!=0)
				lbath+=SGN(i);
			lbath=RETURNLIMIT(lbath,1,NROWS-1-BATH);
		}
	}
	else
	{
		updaterand();
		updaterand();
		updaterand();
		updaterand();
		if ((rand & 15)<2)
		{
			i=ypos+ydir-(rbath+BATH/2);
			if (i!=0)
				rbath+=SGN(i);
			rbath=RETURNLIMIT(rbath,1,NROWS-1-BATH);
		}
	}

	if((framecount%10)==0)
	{
		if (pongpause>0)
			pongpause--;
		else
		{
			xpos+=xdir;
			ypos+=ydir;
			if ((ypos<1)&&(xpos>=COURTL)&&(xpos<=COURTR))
				{ ypos=2; ydir=-ydir; }
			if ((ypos > NROWS-2) && (xpos>=COURTL) && (xpos<=COURTR))
				{ypos = NROWS-3; ydir=-ydir; }
			if (xpos==COURTL)
			{ 	if((ypos>=lbath)&&(ypos<=(lbath+BATH-1)))
					{ xpos=COURTL+1; xdir=-xdir; }
				else
						rscore=(rscore+1)%10;
			}
			if (xpos==COURTR)
			{ 	if((ypos>=rbath)&&(ypos<=(rbath+BATH-1)))
					{ xpos=COURTR-1; xdir=-xdir; }
				else
						lscore=(lscore+1)%10;
			}
			if ((xpos<0) || (xpos>NCOLS-1) || (ypos<0) || (ypos>NROWS-1) )
				 initpongball();
		}
	}
	plot(xpos,ypos);
	drawvline(COURTL, lbath, lbath+BATH-1);
	drawvline(COURTR, rbath, rbath+BATH-1);
	print(SMALLINDEX+lscore,0,1);
	print(SMALLINDEX+rscore,COURTR+2,1);
	next_buf=1-next_buf;
	if ((lscore>3)||(rscore>3))
		cyclefinished=1;
}

void initinvaders(void)
{
	invadervar=NCOLS;
	invaderframecount=0;
	frames=0;
	invaderthresh=40;
}

void iterateinvaders(void)
{
	int8_t t;
	write_buf=1-next_buf;
	clearbuf();
	if ((invadervar&1)==0)
	{
		print(INVADERINDEX,invadervar,1);
		print(INVADERINDEX+1,invadervar+8,1);
		print(INVADERINDEX,invadervar+18,1);
		print(INVADERINDEX+1,invadervar+26,1);
	}
	else
	{
		print(INVADERINDEX+2,invadervar,1);
		print(INVADERINDEX+3,invadervar+8,1);
		print(INVADERINDEX+2,invadervar+18,1);
		print(INVADERINDEX+3,invadervar+26,1);
	}
	next_buf=1-next_buf;
	t=40-2*analoglevel;
	if (t<10)
		t=10;
	if (t<invaderthresh)
		invaderthresh=t;
	invaderframecount++;
	if (invaderframecount>=invaderthresh)
		{
			invadervar=invadervar-1;
			invaderframecount=0;
			if (invadervar==-11)
				invadervar=7;
			invaderthresh=40;
		}
	frames++;
	if (frames>3000)
		cyclefinished=1;
}

void initrandom(void)
{
	cyclefinished=0;
	submode=0;
	initstring();
	readstring(0,0);
	snprintf(&(display_string[0][0]),DISPLAYSTRINGLEN,"rRr   ");
}


void initmode(BYTE m)
{
	switch (m) {
		case TEXT0MODE:
		case TEXT1MODE:
		case TEXT2MODE:
		case TEXT3MODE:
		case TEXT4MODE:
		case TEXT5MODE:
			initstring();
			readstring(m-TEXT0MODE,1);
		break;
		case CYCLICTEXTMODE:
			initstring();
			readstring(0,0);
			snprintf(&(display_string[0][0]),DISPLAYSTRINGLEN,"cCc   ");
		break;
		case ANALOGBOX:
			analogon();
			frames=0;
		break;
		case PONG:
			analogoff();
			initpong();
		break;
		case INVADERS:
			analogon();
			initinvaders();
		break;
		case RANDOM:
			initrandom();
		break;
	}
}

void iteratemode(BYTE m)
{
	switch (m) {
		case TEXT0MODE:
		case TEXT1MODE:
		case TEXT2MODE:
		case TEXT3MODE:
		case TEXT4MODE:
		case TEXT5MODE:
			iteratetextscroll();
		break;
		case CYCLICTEXTMODE: iteratetextscroll();
		break;
		case ANALOGBOX: iteratesoundsquare();
		break;
		case PONG: iteratepong();
		break;
		case INVADERS: iterateinvaders();
		break;
		case RANDOM:
			iteratemode(submode);
		break;
	}
}



int main (void)
{

	BYTE oldframe;
	BYTE switches;
	BYTE oldmode;

	
	mode =0; 
	oldmode =0;
	
	analogontext=1;
	
    ioinit ();
	initstring();
	readstring(0,1);
	
    /* loop forever, the interrupts are doing the rest */


	*(plong_t) buf[0][0]=0;
	*(plong_t) buf[1][0]=0;
	
    for (;;) 
	{
		oldframe=framecount;

		switches = testsw();
		if (switches&(SW1_DOWN|SW2_DOWN|SW3_DOWN|SW1_REPEAT|SW2_REPEAT))
		{
			if (mode!=MENU)
			{
				if (switches==SW1_DOWN)
					{ mode=(mode + 1) % NCYCLEMODES; }
				if (switches==SW2_DOWN)
					{ mode= (mode == 0) ? NCYCLEMODES-1 : mode-1; }
				if (switches==SW3_DOWN)
				{
					oldmode = mode;
					mode = MENU;
					if (oldmode > TEXT5MODE)
						submode=0;
					else
					{
						submode=medit;
						editstringno=oldmode;
					}
					subsubmode=0;
					initmenu();
					printmenu();
				}
			}
			else /* mode == menu */
			{
				if (menuinput(switches))
					mode=oldmode;
			}
			initmode(mode);
		}
		
		if ((mode==RANDOM)&&(cyclefinished))
		{
			updaterand();
			updaterand();
			updaterand();
			submode=(submode+(rand&7))%NORDINARYMODES;
			cyclefinished=0;
			initmode(submode);
		}
		
		iteratemode(mode);
		
		
		while(framecount==oldframe)
			sleep_mode();
		if (framecount >= FPS)
			framecount = 0;
	}
	return(0);
}
