
#include <stdint.h>
#include <math.h>
#include <stdlib.h> // Used for rand()
#include <avr/interrupt.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include "main.h"
#include "midi.h"


// Number of waveforms
#define NWAVES 5

// Number of LFO modes
#define NLFOMODES 4

// Number of CV modes
#define NCVMODES 5

#define BENDSCALELO (2.0)
#define BENDSCALEHI (36.0)
#define LOWCVCENTRE (59.0)
#define FULLSCALE (47.0)
#define FULLCENTRE (59.0)


#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))

// Forumula to convert MIDI note number to frequency is
// F=440 * (2^(1/12)) ^ (note - 69)  Assuming A=440Hz
// Then to convert to increment value
// increment = (2^32) * f/ (F_CPU)
//
// Crunching numerically and optimising we get

#define INCFROMNOTE(n) (2194.71 * exp(0.057762 * ((float) n) ) )

#define LFOLOWNOTE (-36) // Midi note equivalence to lowest LFO frequency (1Hz)
#define LFONOTERANGE (92) // Range of LFO tuning in semitones/MIDI notes (set for 1-200Hz)

#define MIDINOTEONFLAG 1
#define MIDINOTEOFFFLAG 2

#define CV1INPUT 2
#define CV2INPUT 0

#define GATEIN (PIND & _BV(PD4))

//global data
asm(".global wave"); //if want seen by others
asm(".data");
// The next line is in-theory necessary to force "wave" on to a page (256 byte) boundary
// but as "wave" is allocated at the start of ".data" it is aligned to boundary
// in any case. Including the next line appears to mess-up the allocation of
// the ".bss" section wasting memory
//
// After major changes I suggest you check you ".sym" file to see if "wave"
// is on a page boundary
//
//asm(".balign 256"); // Align to page boundary
asm("wave: .fill 512");
asm(".global serbuf");
asm( BUFASM );
asm(".global accumulator");
asm("accumulator: .fill 8");
asm(".global increment");
asm("increment: .fill 8");
asm(".global bufptr");
asm("bufptr: .fill 1");
asm(".global buflen");
asm("buflen: .fill 1");
asm(".text");



typedef struct {
	uint8_t wave1;
	uint8_t wave2;
	uint8_t lfomode;
	uint8_t cvmode;
	uint8_t gate;
} machinestate;


extern uint8_t  wave[2][256];// __attribute__ ((aligned (256)));

machinestate oldstate = { 0, 0, 1, 0, 0};
machinestate newstate = { 0, 0, 1, 0, 0};
uint8_t  midisignal=0;
uint8_t  midinotestate=0;
extern volatile uint32_t accumulator[2];
extern volatile uint32_t increment[2];
volatile uint8_t lfooneshot=1; // if starting in "norm" mode
volatile uint8_t lforeset=1;

// MIDI storage globals
byte	lastnote;
byte	lastchannel;
float lastbend=0;
float lastlfo=LFOLOWNOTE; // Last LFO value received from MIDI (via "modulation" control)

// Systen state variables
float cv[2]={0};
int8_t controlvalue[4]={0,0,1,0};
int8_t ledcolumn=0;
int8_t debounce[4]={100};





void setup(void)
{
	MCUCR &= _BV(PUD); // Enable pull-up resistors
	DDRD|=_BV(PD5); // Output for OC0B
	DDRB|=_BV(PB3); // Output for gate
	DDRD|=_BV(PD3); // Set output on OC2B
	DDRD|=_BV(PD6); // Set output on OC0A
	
	// Set up Timer 0 to provide signal f output
	
	// Set fast PWM mode
	TCCR0A |= _BV(WGM01) | _BV(WGM00);
    TCCR0B &= ~_BV(WGM02);
	
	// Do inverting PWM on pin OC0A
    // On the Arduino this is pin 6.
	TCCR0A = TCCR0A | _BV(COM0A1) | _BV(COM0A0);
   
    // Alternative for non-inverting PWM
	// TCCR0A = (TCCR0A | _BV(COM0A1)) & ~_BV(COM0A0);
	
	// Do Inverting PWM on pin OC0B
	// On the Arduino this is pin 5
	TCCR0A = TCCR0A | _BV(COM0B1) | _BV(COM0B0);
	
	// Alternative for non-inverting PWM:
	// TCCR0A = (TCCR0A | _BV(COM0B1)) & ~_BV(COM0B0);
	
	// No prescaler (p.158)
    TCCR0B = (TCCR0B & ~(_BV(CS02) | _BV(CS01))) | _BV(CS00);

    // Set up Timer 2 to do pulse width modulation 

    // Use internal clock (datasheet p.160)
    ASSR &= ~( _BV(AS2));

    // Set fast PWM mode  (p.157)
    TCCR2A |= _BV(WGM21) | _BV(WGM20);
    TCCR2B &= ~_BV(WGM22);

	
	// No non-inverting PWM on OC2B
	// On the Arduino this is PIN3 (PD3)
    TCCR2A = TCCR2A | _BV(COM2B1) | _BV(COM2B0);
	
	// Alernative for inverting PWM
	// TCCR2A = (TCCR2A | _BV(COM2B1)) & ~_BV(COM2B0);
	
    // No prescaler (p.158)
    TCCR2B = (TCCR2B & ~(_BV(CS22) | _BV(CS21))) | _BV(CS20);
	
	
	// Set up AVCC reference
	 ADMUX = _BV(REFS0); // 5V vref
	
	// Set up free running ADC with max prescaling
	
	ADCSRA=_BV(ADPS2)|_BV(ADPS1)|_BV(ADPS0);
	ADCSRA|=_BV(ADEN);
	ADCSRA|=_BV(ADSC);
	
	// Turn off alanog comparator
	
	ACSR=_BV(ACD);
	
	// Set counter values for timer 0 and timer2 to dephase them
	TCNT0=0;
	TCNT2=128;

	// Set timer 1 with 8 precaling to count to OCR1A
	// Timer 1 is polled to refresh LED matrix
//	TCCR1B=_BV(WGM12)|_BV(CS11); 
//	OCR1A=2500;

    cli();
  
		TIMSK0=_BV(TOIE0);

    sei();
	
}

void setosc1(float n) {
	uint32_t tempinc;
	tempinc=INCFROMNOTE(n);
	cli();
	increment[0]=tempinc;
	sei();
	OCR2B=CLAMP((n-12)*2.586,0,255); // Set Log F output
}

static inline void stoposc1() {
	cli();
	increment[0]=0;
	sei();
}

void setosc2(float n, uint8_t resetflag) {
	uint32_t tempinc;
	tempinc=INCFROMNOTE(n);
	if (! resetflag) {
		cli();
		increment[1]=tempinc;
		sei();
	} else {
		cli();
		increment[1]=tempinc;
		accumulator[1]=0;
		sei();
	}
} 

static inline void stoposc2() {
	cli();
	increment[1]=0;
	sei();
}

void HandleNoteOff(byte channel, byte pitch, byte velocity) {
	if ((channel==lastchannel) && (pitch==lastnote)) {
		midisignal|=MIDINOTEOFFFLAG;
		midinotestate =0;
	}
}

void HandleNoteOn(byte channel, byte pitch, byte velocity) {
	if (velocity==0) {
		HandleNoteOff(channel, pitch, velocity);
	} else {
		midisignal|=MIDINOTEONFLAG;
		lastchannel=channel;
		lastnote=pitch;
		midinotestate=1;
	}
}


void PitchBendProc(byte channel, int bend) {
	if (channel==lastchannel) {
		lastbend=((float)bend)/2500.0;
	}
}

void ControlChangeProc(byte channel, byte control, byte value) {
	if (control==1) {
		lastlfo=LFOLOWNOTE+LFONOTERANGE*(float)value/127.0;
		}
}


static inline void readcv(uint8_t knob, int16_t sample) {
	cv[knob]=((float)(sample-0x1ff))/((float)0x1ff);
}



void setledrow(int8_t r) {
// Test
//	DDRD|=_BV(PD7);
//	PORTD&=~_BV(PD7);
	switch(r) {
		case 0:
			DDRD|=_BV(PD7);
			PORTD&=~_BV(PD7);
		break;
		case 1:
			DDRC|=_BV(PC1);
			PORTC&=~_BV(PC1);
		break;
		case 2:
			DDRD|=_BV(PD2);
			PORTD&=~_BV(PD2);
		break;
		case 3:
			DDRD|=_BV(PD1);
			PORTD&=~_BV(PD1);
		break;
		case 4:
			DDRC|=_BV(PC5);
			PORTC&=~_BV(PC5);
		break;
		default:;
	}
}

int8_t processbutton(int8_t bitstate, int8_t button, int8_t nvalues) {
	if (!bitstate) {
		if (debounce[button]>2) {
			controlvalue[button]=(controlvalue[button]+1)%nvalues;
			debounce[button]=0;
			return true;
			}
	} else {
		if (debounce[button]<100)
			debounce[button]+=1;
	}
	return false;
}

void updatematrix() {
	if (ledcolumn==4) {
		if (processbutton((PINB & _BV(PB0)),0,NWAVES))
			newstate.wave1=controlvalue[0];
		if (processbutton((PINC & _BV(PC3)),1,NWAVES))
			newstate.wave2=controlvalue[1];
		if (processbutton((PINC & _BV(PC4)),2,NLFOMODES))
			newstate.lfomode=controlvalue[2];
		if (processbutton((PINB & _BV(PB1)),3,NCVMODES))
			newstate.cvmode=controlvalue[3];
	}
	ledcolumn=(ledcolumn+1)%5;
	DDRB&=~(_BV(PB0)|_BV(PB1)|_BV(PB2));
	DDRC&=~(_BV(PC1)|_BV(PC3)|_BV(PC4)|_BV(PC5));
	DDRD&=~(_BV(PD1)|_BV(PD2)|_BV(PD7));
	
	PORTB&=~(_BV(PB0)|_BV(PB1)|_BV(PB2));
	PORTC&=~(_BV(PC1)|_BV(PC3)|_BV(PC4)|_BV(PC5));
	PORTD&=~(_BV(PD1)|_BV(PD2)|_BV(PD7));

	
	
	
	switch(ledcolumn) {
		case 0:
			DDRB|=_BV(PB0);
			PORTB|=_BV(PB0);
			setledrow(controlvalue[ledcolumn]);
		break;
		case 1:
			DDRC|=_BV(PC3);
			PORTC|=_BV(PC3);
			setledrow(controlvalue[ledcolumn]);
		break;
		case 2:
			DDRC|=_BV(PC4);
			PORTC|=_BV(PC4);
			setledrow(controlvalue[ledcolumn]);
		break;
		case 3:
			DDRB|=_BV(PB1);
			PORTB|=_BV(PB1);
			setledrow(controlvalue[ledcolumn]);
		break;
		case 4:
			DDRB|=_BV(PB2); // Set Row 6 output low
			PORTB|=_BV(PB0)|_BV(PB1);
			PORTC|=_BV(PC3)|_BV(PC4);
		break;
		default:;
	}
	

}

void polladc(void) {
	int16_t sample;
	uint8_t lastmux;
	if (!(ADCSRA & _BV(ADSC))) {
		lastmux=ADMUX & 0xf;
		if (lastmux==2)
			ADMUX=(ADMUX & 0xf0);
		else
			ADMUX=(ADMUX & 0xf0)|2;
		sample=ADCL;
		sample|=(ADCH<<8);
		switch (lastmux) {
			case CV1INPUT:
				readcv(0, sample);
				break;
			case CV2INPUT:
				readcv(1, sample);
				break;
		} // switch
		updatematrix();
		ADCSRA|=_BV(ADSC); // start next conversion
	} // if
	
}

void dooften(void) {
	polladc();
	MIDI.read();
}

void setwave(uint8_t table[], int8_t w) {
	int16_t i;
	for (i=0;i<256;i++) {
		dooften();
		switch (w) {
			case 0:
				table[i]=255-i;
				break;
			case 1:
				if (i<=127)
					table[i]=255;
				else
					table[i]=0;
				break;
			case 2:
				table[i]=128+127*sin(2.0*3.14129*i/256.0);
				break;
			case 3:
				if (i<=127)
					table[i]=2*i;
				else
					table[i]=255-2*(i-128);
				break;
			case 4:
				table[i]=i;
				break;
		}
	}
}

static inline void gateon(void) {
	PORTB |= _BV(PB3);
}

static inline void gateoff(void) {
	PORTB &= ~ _BV(PB3);
}

float calcoscfreq(uint8_t mode, uint8_t cvindex) {
	switch (mode) {
		case 0:	return (lastbend+lastnote);
		break;
		case 1:	return (BENDSCALELO*cv[cvindex]+lastbend+lastnote);
		break;
		case 2:	return (BENDSCALEHI*cv[cvindex]+lastbend+lastnote);
		break;
		case 3: return (BENDSCALEHI*cv[cvindex]+LOWCVCENTRE);
		break;
		case 4: return (FULLSCALE*cv[cvindex]+FULLCENTRE);
		break;
		default: 
		return (LOWCVCENTRE);
	}
}


void updatesound(void) {
	uint8_t resetlfoflag=0; // Set true if we are going to force the LFO to restart
	float	osc1freq, osc2freq;
	if (midisignal & MIDINOTEONFLAG) // Start of a MIDI note?
		gateoff(); // Force external gate signal low immediately before we start a new note
		
	if ((midisignal & MIDINOTEONFLAG) || ((! oldstate.gate) && newstate.gate) ) { // Starting a new note?
		resetlfoflag |= (newstate.lfomode == 1) || (newstate.lfomode==2);
		if (newstate.lfomode==1)
			lfooneshot=0; // Make sure LFO repeats in Norm mode as it might have been stopped by the end of a note
	}
	if (newstate.wave1 != oldstate.wave1)
		setwave(wave[0],newstate.wave1);
	if (newstate.wave2 != oldstate.wave2)
		setwave(wave[1],newstate.wave2);
	if (newstate.lfomode != oldstate.lfomode) {
		if (newstate.lfomode == 2)
			lfooneshot=1;
		else {
			if ((newstate.lfomode!=1) || midinotestate || newstate.gate)
				lfooneshot=0;
			else lfooneshot=1;
			resetlfoflag=1;
		};
	};
					
	osc1freq=calcoscfreq(newstate.cvmode, 0);
	
	if (newstate.lfomode==3)
		osc2freq=calcoscfreq(newstate.cvmode,1);
	else
		switch (newstate.cvmode) {
			case 0: osc2freq=lastlfo;
			break;
			case 1:
			case 2: osc2freq=LFONOTERANGE/2.0+lastlfo+cv[1]*LFONOTERANGE/2.0;
			break;
			default:
				osc2freq=LFOLOWNOTE+LFONOTERANGE/2.0+cv[1]*LFONOTERANGE/2.0;
		}

	if (midinotestate)
		gateon();
	else
		gateoff();
	if (midinotestate || newstate.gate) { // Note is playing
		setosc1(osc1freq);
	}
	else {
		stoposc1();
	}
	
	if (newstate.lfomode!=3) {
		if ((newstate.lfomode !=2) || resetlfoflag)
			setosc2(osc2freq, resetlfoflag); // LFO mode 0 or 1 or (2 with new note)
		if ((newstate.lfomode==1) && (
				((midisignal & MIDINOTEOFFFLAG) && (!newstate.gate)) || 
				((!midinotestate) && oldstate.gate && (!newstate.gate))
				)
			)
				lfooneshot=1; // In norm mode stop oscillator at end of note
	} else { // LFO mode = 3 - Second oscillator mode
		if (midinotestate || newstate.gate)
			setosc2(osc2freq,0);
		else
			stoposc2();
	}
}

int main(void){
	setup();
	MIDI.begin(MIDI_CHANNEL_OMNI);
	MIDI.setHandleNoteOn(HandleNoteOn);
	MIDI.setHandleNoteOff(HandleNoteOff);
	MIDI.setHandlePitchBend(PitchBendProc);
	MIDI.setHandleControlChange(ControlChangeProc);
	setwave(wave[0],0);
	setwave(wave[1],0);
	// Testing section
	//setosc1(60);
	//midinotestate=1;
	//setosc2(10,0);
	// End testing section
	
	while (1) {
		polladc();
		MIDI.read();
		newstate.gate=GATEIN;
		updatesound();
		oldstate=newstate;
		midisignal=0;
	}

	
}
